/*
 * Decompiled with CFR 0.152.
 */
package dev.ftb.mods.ftbbackups;

import com.google.gson.JsonElement;
import com.mojang.serialization.DynamicOps;
import com.mojang.serialization.JsonOps;
import dev.ftb.mods.ftbbackups.BackupStatus;
import dev.ftb.mods.ftbbackups.BackupUtils;
import dev.ftb.mods.ftbbackups.api.Backup;
import dev.ftb.mods.ftbbackups.api.IArchivalPlugin;
import dev.ftb.mods.ftbbackups.api.event.BackupEvent;
import dev.ftb.mods.ftbbackups.archival.ArchivePluginManager;
import dev.ftb.mods.ftbbackups.config.FTBBackupsServerConfig;
import dev.ftb.mods.ftbbackups.net.BackupProgressPacket;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import net.minecraft.ChatFormatting;
import net.minecraft.Util;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.network.chat.TextColor;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.level.storage.LevelResource;
import net.neoforged.bus.api.Event;
import net.neoforged.fml.loading.FMLPaths;
import net.neoforged.neoforge.common.NeoForge;
import org.apache.commons.lang3.mutable.MutableInt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Backups {
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd-HH-mm-ss");
    public static final Logger LOGGER = LoggerFactory.getLogger(Backups.class);
    public static final String BACKUPS_JSON_FILE = "backups.json";
    private static Backups clientInstance = null;
    private static Backups serverInstance = null;
    private final List<Backup> backups = new ArrayList<Backup>();
    public Path backupsFolder;
    public long nextBackupTime = -1L;
    public BackupStatus backupStatus;
    public boolean printFiles = false;
    private final AtomicInteger currentFileIndex = new AtomicInteger(0);
    private final AtomicInteger totalFileCount = new AtomicInteger(0);
    private int prevFileIndex = 0;
    private String currentFileName = "";
    public boolean playersOnlineSinceLastBackup = false;

    public static Backups getClientInstance() {
        if (clientInstance == null) {
            clientInstance = new Backups();
        }
        clientInstance.loadIndexFile();
        return clientInstance;
    }

    public static void initServerInstance() {
        serverInstance = new Backups();
    }

    public static Backups getServerInstance() {
        if (serverInstance == null) {
            Backups.initServerInstance();
        }
        return serverInstance;
    }

    private Backups() {
        this.backupsFolder = FTBBackupsServerConfig.getBackupFolder();
        try {
            Files.createDirectories(this.backupsFolder, new FileAttribute[0]);
            this.backupsFolder = this.backupsFolder.toRealPath(new LinkOption[0]);
        }
        catch (Exception ignored) {
            LOGGER.error("Failed to create backups folder: {}", (Object)this.backupsFolder);
        }
        this.backupStatus = BackupStatus.NONE;
        this.loadIndexFile();
        LOGGER.info("Backups will be written to: {}", (Object)this.backupsFolder);
    }

    public void loadIndexFile() {
        this.backups.clear();
        Path backupsIndex = Backups.backupsIndexPath();
        JsonElement element = BackupUtils.readJson(backupsIndex);
        Backup.LIST_CODEC.parse((DynamicOps)JsonOps.INSTANCE, (Object)element).resultOrPartial(err -> LOGGER.warn("can't parse backups index {}", (Object)backupsIndex)).ifPresent(this.backups::addAll);
    }

    private static Path backupsIndexPath() {
        return FMLPaths.GAMEDIR.get().resolve("ftbbackups3").resolve(BACKUPS_JSON_FILE);
    }

    public void tick(MinecraftServer server, long now) {
        if (this.nextBackupTime < 0L) {
            this.nextBackupTime = System.currentTimeMillis() + FTBBackupsServerConfig.getBackupTimerMillis();
        }
        if (!(this.nextBackupTime <= 0L || this.nextBackupTime > now || ((Boolean)FTBBackupsServerConfig.ONLY_IF_PLAYERS_ONLINE.get()).booleanValue() && !this.playersOnlineSinceLastBackup && server.getPlayerList().getPlayers().isEmpty())) {
            this.playersOnlineSinceLastBackup = false;
            this.run(server, true, (Component)Component.translatable((String)"ftbbackups3.lang.server"), "");
        }
        if (this.backupStatus.isDone()) {
            this.backupStatus = BackupStatus.NONE;
            for (ServerLevel level : server.getAllLevels()) {
                if (level == null) continue;
                level.noSave = false;
            }
            if (!((Boolean)FTBBackupsServerConfig.SILENT.get()).booleanValue()) {
                BackupProgressPacket.sendProgress(server, BackupProgressPacket.complete());
            }
            this.currentFileIndex.set(0);
            this.totalFileCount.set(0);
        } else if (this.backupStatus.isRunning() && this.printFiles) {
            int curIdx = this.currentFileIndex.get();
            if (curIdx == 0 || curIdx == this.totalFileCount.get() - 1) {
                LOGGER.info("[{} | {}%]: {}", new Object[]{this.currentFileIndex, (int)((double)curIdx / (double)this.totalFileCount.get() * 100.0), this.currentFileName});
            }
            if (!((Boolean)FTBBackupsServerConfig.SILENT.get()).booleanValue() && this.prevFileIndex != this.currentFileIndex.get()) {
                BackupProgressPacket.sendProgress(server, BackupProgressPacket.update());
                this.prevFileIndex = this.currentFileIndex.get();
            }
        }
    }

    public void notifyAll(MinecraftServer server, Component component, boolean error) {
        component = component.plainCopy();
        component.getStyle().withColor(TextColor.fromLegacyFormat((ChatFormatting)(error ? ChatFormatting.DARK_RED : ChatFormatting.LIGHT_PURPLE)));
        server.getPlayerList().broadcastSystemMessage(component, true);
    }

    public boolean run(MinecraftServer server, boolean auto, Component name, String customName) {
        if (this.backupStatus.isRunningOrDone()) {
            return false;
        }
        if (auto && !((Boolean)FTBBackupsServerConfig.AUTO.get()).booleanValue()) {
            return false;
        }
        this.notifyAll(server, (Component)Component.translatable((String)"ftbbackups3.lang.start", (Object[])new Object[]{name}), false);
        LOGGER.info("backup starting: {}", (Object)name.getString());
        this.nextBackupTime = System.currentTimeMillis() + FTBBackupsServerConfig.getBackupTimerMillis();
        for (ServerLevel level : server.getAllLevels()) {
            if (level == null) continue;
            level.noSave = true;
        }
        this.backupStatus = BackupStatus.RUNNING;
        server.getPlayerList().saveAll();
        CompletableFuture.runAsync(() -> {
            try {
                this.createBackup(server, customName);
            }
            catch (Exception ex) {
                LOGGER.error("backup creation failed: {} -> {}", ex.getClass(), (Object)ex.getMessage());
                server.execute(() -> this.notifyAll(server, (Component)Component.translatable((String)"ftbbackups3.lang.saving_failed"), true));
            }
        }).thenRun(() -> {
            this.backupStatus = BackupStatus.DONE;
        });
        return true;
    }

    private void createBackup(MinecraftServer server, String customName) {
        Path absoluteWorldPath = server.getWorldPath(LevelResource.ROOT).toAbsolutePath();
        IArchivalPlugin archivalPlugin = ArchivePluginManager.serverInstance().getPlugin(FTBBackupsServerConfig.archivalPlugin());
        if (archivalPlugin == null) {
            LOGGER.error("bad archive plugin setting in server config: {}! backup skipped", (Object)FTBBackupsServerConfig.archivalPlugin());
            return;
        }
        String backupFileName = archivalPlugin.addFileExtension(customName.isEmpty() ? DATE_TIME_FORMATTER.format(LocalDateTime.now()) : customName);
        long now = Util.getEpochMillis();
        boolean success = false;
        Exception error = null;
        long archiveSize = 0L;
        MutableInt fileCount = new MutableInt(0);
        try {
            Map<Path, String> manifest = this.gatherFiles(server);
            fileCount.setValue(manifest.size());
            for (Map.Entry<Path, String> entry : manifest.entrySet()) {
                archiveSize += Files.size(entry.getKey());
            }
            this.doArchiveCleanup(archiveSize);
            this.totalFileCount.set(manifest.size());
            LOGGER.info("Backing up {} files...", (Object)this.totalFileCount);
            this.printFiles = true;
            BackupProgressPacket.sendProgress(server, BackupProgressPacket.start());
            this.currentFileIndex.set(0);
            Path archiveDest = this.backupsFolder.resolve(backupFileName);
            long start = Util.getEpochMillis();
            archiveSize = archivalPlugin.createArchive(new BackupContext(manifest, LOGGER, archiveDest, (Integer)FTBBackupsServerConfig.COMPRESSION_LEVEL.get(), this));
            LOGGER.info("Created archive {} from {} in {} ms ({})!", new Object[]{archiveDest, absoluteWorldPath, Util.getEpochMillis() - start, BackupUtils.formatSizeString(archiveSize)});
            success = true;
        }
        catch (Exception ex) {
            if (!((Boolean)FTBBackupsServerConfig.SILENT.get()).booleanValue()) {
                String errorName = ex.getClass().getName();
                this.notifyAll(server, (Component)Component.translatable((String)"ftbbackups3.lang.fail", (Object[])new Object[]{errorName}), true);
            }
            LOGGER.error("Backup to {} failed: {} -> {}", new Object[]{backupFileName, ex.getClass(), ex.getMessage()});
            error = ex;
        }
        this.printFiles = false;
        Backup backup = new Backup(now, archivalPlugin.getId(), backupFileName, server.getWorldData().getLevelName(), this.getLastIndex() + 1, success, archiveSize, fileCount.getValue());
        this.backups.add(backup);
        Backup.LIST_CODEC.encodeStart((DynamicOps)JsonOps.INSTANCE, this.backups).ifSuccess(json -> BackupUtils.writeJson(Backups.backupsIndexPath(), json, true));
        NeoForge.EVENT_BUS.post((Event)new BackupEvent.Post(backup, error));
        if (error == null && !((Boolean)FTBBackupsServerConfig.SILENT.get()).booleanValue()) {
            this.processBackupResults(server, now, archiveSize);
        }
    }

    private Map<Path, String> gatherFiles(MinecraftServer server) throws IOException {
        final LinkedHashMap<Path, String> res = new LinkedHashMap<Path, String>();
        final Path instanceDir = server.getServerDirectory().toRealPath(new LinkOption[0]);
        Consumer<Path> extras = path -> {
            try {
                Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(this){

                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                        if (file.isAbsolute()) {
                            LOGGER.error("ignoring absolute file {} in extras!", (Object)file);
                        } else if (Backups.notExcludedByConfig(file)) {
                            res.put(file.toRealPath(new LinkOption[0]), file.toString());
                        }
                        return FileVisitResult.CONTINUE;
                    }
                });
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        };
        for (String s : (List)FTBBackupsServerConfig.EXTRA_FILES.get()) {
            extras.accept(Path.of(s, new String[0]));
        }
        NeoForge.EVENT_BUS.post((Event)new BackupEvent.Pre(extras));
        Files.walkFileTree(server.getWorldPath(LevelResource.ROOT).toAbsolutePath(), (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(this){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (Files.isReadable(file) && Backups.notExcludedByConfig(file)) {
                    res.put(file.toRealPath(new LinkOption[0]), instanceDir.relativize(file).toString());
                }
                return FileVisitResult.CONTINUE;
            }
        });
        return res;
    }

    private static boolean notExcludedByConfig(Path file) {
        return ((List)FTBBackupsServerConfig.EXCLUSION_MATCHERS.get()).stream().noneMatch(m -> m.matches(file.getFileName()));
    }

    private void doArchiveCleanup(long fileSize) {
        if (!this.backups.isEmpty()) {
            this.backups.sort(null);
            int backupsToKeep = (Integer)FTBBackupsServerConfig.BACKUPS_TO_KEEP.get() - 1;
            if (backupsToKeep > 0 && this.backups.size() > backupsToKeep) {
                while (this.backups.size() > backupsToKeep) {
                    Backup backup = this.backups.removeFirst();
                    backup.deleteFiles(this.backupsFolder);
                }
            }
            long totalSize = this.backups.stream().mapToLong(Backup::size).sum();
            if (fileSize > 0L) {
                long freeSpace = Math.min((Long)FTBBackupsServerConfig.MAX_TOTAL_SIZE.get(), this.backupsFolder.toFile().getFreeSpace());
                while (totalSize + fileSize > freeSpace && !this.backups.isEmpty()) {
                    Backup backup = this.backups.removeFirst();
                    if (!backup.deleteFiles(this.backupsFolder)) continue;
                    totalSize -= backup.size();
                }
            }
        }
    }

    private void processBackupResults(MinecraftServer server, long backupTime, long archiveSize) {
        MutableComponent component;
        Duration d = Duration.ZERO.plusMillis(Util.getEpochMillis() - backupTime);
        String timeString = String.format("%d.%03ds", d.toSeconds(), d.toMillisPart());
        if (((Boolean)FTBBackupsServerConfig.DISPLAY_FILE_SIZE.get()).booleanValue()) {
            String sizeT;
            long totalSize = this.backups.stream().mapToLong(Backup::size).sum();
            String sizeB = BackupUtils.formatSizeString(archiveSize);
            String sizeString = sizeB.equals(sizeT = BackupUtils.formatSizeString(totalSize)) ? sizeB : sizeB + " | " + sizeT;
            component = Component.translatable((String)"ftbbackups3.lang.end_2", (Object[])new Object[]{timeString, sizeString});
        } else {
            component = Component.translatable((String)"ftbbackups3.lang.end_1", (Object[])new Object[]{timeString});
        }
        component.getStyle().withColor(TextColor.fromLegacyFormat((ChatFormatting)ChatFormatting.LIGHT_PURPLE));
        server.getPlayerList().broadcastSystemMessage((Component)component, true);
    }

    public long totalBackupSize() {
        return this.backups.stream().mapToLong(Backup::size).sum();
    }

    private int getLastIndex() {
        int i = 0;
        for (Backup b : this.backups) {
            i = Math.max(i, b.index());
        }
        return i;
    }

    public int getCurrentFileIndex() {
        return this.currentFileIndex.get();
    }

    public int getTotalFileCount() {
        return this.totalFileCount.get();
    }

    public Collection<Backup> backups() {
        return Collections.unmodifiableCollection(this.backups);
    }

    private record BackupContext(Map<Path, String> manifest, Logger logger, Path archivePath, int compressionLevel, Backups backups) implements IArchivalPlugin.ArchivalContext
    {
        @Override
        public synchronized void notifyProcessingFile(String filename) {
            this.backups.currentFileName = filename;
            this.backups.currentFileIndex.incrementAndGet();
        }
    }
}

